
#include "VulkanAppContext.h"
#include "vkaUtils.h"
#include "VKSFile.h"

#include<string>


#include"VkeCreateUtils.h"
#include"VkeRenderer.h"


#include"VkeGameRendererDynamic.h"

#include"VulkanDeviceContext.h"
#include<iostream>

#ifndef INIT_COMMAND_ID
#define INIT_COMMAND_ID 1
#endif

#define DEFAULT_SCENE_ID 1

#define RENDERER vkeGameRendererDynamic



VulkanAppContext *VulkanAppContext::GetInstance(){
	static VulkanAppContext *instance = NULL;

	if (!instance){
		instance = new VulkanAppContext();
	}

	return instance;
}

VulkanAppContext::VulkanAppContext():
m_width(1024),
m_height(768),
m_ready(false),
m_current_frame(0),
m_rotor_node(NULL)
{
}

void VulkanAppContext::loadVKSScene(std::string &inFileName){
	VKSFile vkFile;

	vkFile.outputFile = inFileName;


	readVKSFile(&vkFile);

	/*
		Unpack meshes
	*/

	uint32_t meshCnt = vkFile.header.meshCount;

#if USE_SINGLE_VBO

	uint32_t vtxStoreSize = vkFile.vertexCount * 10 * sizeof(float);
	uint32_t idxStoreSize = vkFile.indexCount * sizeof(uint32_t);

	m_global_vbo.initBackingStore(vtxStoreSize);
	m_global_ibo.initBackingStore(idxStoreSize);

	float *vData = m_global_vbo.getBackingStore();
	memcpy(vData, vkFile.vertices.data(), vtxStoreSize);
	uint32_t *iData = m_global_ibo.getBackingStore();
	memcpy(iData, vkFile.indices.data(), idxStoreSize);


	m_global_vbo.initVKBufferData();
	m_global_ibo.initVKBufferData();



#endif


	uint32_t animCount = vkFile.header.animationCount;


	if (animCount >= 1){
		//importing the first animation only at the moment.
		VKSAnimationRecord animation = vkFile.animations[0];
		uint32_t nCnt = animation.nodecount;

		for (uint32_t n = 0; n < nCnt; ++n){
			VKSAnimationNodeRecord nodeAnim = vkFile.animationNodes[n + animation.firstNode];

			std::string newNodeName(nodeAnim.name);

			VkeAnimationNode *vAnimNode = m_animation.newNode(newNodeName);

			uint32_t keyCount = nodeAnim.positionCount;

			for (uint32_t k = 0; k < keyCount; ++k){
				VKSAnimationKeyRecord key = vkFile.animationKeys[k + nodeAnim.firstPosition];
				vAnimNode->Position().newKey(key.time, key.key);
			}

			keyCount = nodeAnim.rotationCount;

			for (uint32_t k = 0; k < keyCount; ++k){
				VKSAnimationKeyRecord key = vkFile.animationKeys[k + nodeAnim.firstRotation];
				vAnimNode->Rotation().newKey(key.time, key.key);
			}

			keyCount = nodeAnim.scaleCount;

			for (uint32_t k = 0; k < keyCount; ++k){
				VKSAnimationKeyRecord key = vkFile.animationKeys[k + nodeAnim.firstScale];
				vAnimNode->Scale().newKey(key.time, key.key);
			}
		}
	}



	for (uint32_t i = 0; i < meshCnt; ++i){
		VkeMesh *theMesh = m_mesh_data.newMesh(i, &vkFile, &vkFile.meshes[i]);

#if USE_SINGLE_VBO
#else
		theMesh->initVKBuffers();
#endif


		theMesh->setFirstIndex(vkFile.meshes[i].firstIndex);
		theMesh->setFirstVertex(vkFile.meshes[i].firstVertex);

	
		printf("First Index : %d First Vertex : %d.\n", vkFile.meshes[i].firstIndex, vkFile.meshes[i].firstVertex);
	}

	uint32_t matCnt = vkFile.header.materialCount;


	for (uint32_t i = 0; i < matCnt; ++i){
		m_materials.newMaterial(i)->initFromData(&vkFile,&vkFile.materials[i]);
	}
	
	uint32_t nodeCnt = vkFile.header.nodeCount;
	uint32_t nodesProcessed = 0;


	while (nodesProcessed < nodeCnt){
		addVKSNode(&vkFile, nodesProcessed);
	}



	m_node_data.sortByOpacity();

}

void VulkanAppContext::addVKSNode(VKSFile *inFile, uint32_t &inNodesProcessed, Node *parentNode){

	uint32_t nodeID = inNodesProcessed;
	VKSNodeRecord *fileNode = &inFile->nodes[inNodesProcessed++];
	VkeNodeData *vNode;
	Node *node;
	
	uint32_t mshCount = fileNode->meshCount;

	for (uint32_t i = 0; i < mshCount; ++i){

		if (parentNode){
			node = parentNode->newChild(nodeID);
		}
		else{
			node = m_scene_graph->Nodes().newNode(nodeID);
		}

		node->setPosition(
			fileNode->position.x,
			fileNode->position.y,
			fileNode->position.z
			);

		node->setRotation(fileNode->rotation);
		node->setScale(fileNode->scale.x, fileNode->scale.y, fileNode->scale.z);


		m_node_data.newData(node->getID())->updateFromNode(node);
		m_node_data.getData(node->getID())->setMesh(m_mesh_data.getMesh(fileNode->meshIndices[i]));
	}

	std::string nameStr = std::string(fileNode->name);

	VkeAnimationNode *animNode = m_animation.Nodes().getNode(nameStr);
	if (animNode){
		animNode->setNode(m_node_data.getData(node->getID()));
	}

	if (nameStr == "main_rotor_parts02"){
		m_rotor_node = m_node_data.getData(node->getID());
	}

	uint32_t childCount = fileNode->childCount;
	for (uint32_t i = 0; i < childCount; ++i){
		addVKSNode(inFile, inNodesProcessed, node);
	}
}


void VulkanAppContext::initRenderer(){

    RenderContext *rctxt = RenderContext::Get();
    if (!rctxt) return;

	
	/*
		Used to manage the rotation of the 
		Gazelle's rotors.
	*/

	m_rot_y = 0.0f;
	m_current_time = 0.0;

	/*
		Create the renderer.
	*/

    m_renderer = new RENDERER();

    m_scene_graph = rctxt->newScene(DEFAULT_SCENE_ID);

	std::string sceneFile = "chopper_pack32.vks";

	loadVKSScene(sceneFile);





	((RENDERER*)m_renderer)->setNodeData(&m_node_data);
	((RENDERER*)m_renderer)->setMaterialData(&m_materials);
	((RENDERER*)m_renderer)->initIndirectCommands();
	m_renderer->initShaders();

	m_renderer->initLayouts();

	VkCommandBuffer cmd = VK_NULL_HANDLE;
	VulkanDC::Device::Queue::CommandBufferID cmdID = INIT_COMMAND_ID + 300;
	VulkanDC *dc = VulkanDC::Get();
	VulkanDC::Device *device = dc->getDefaultDevice();

	dc->getDefaultQueue()->beginCommandBuffer(cmdID, &cmd, VK_COMMAND_BUFFER_USAGE_ONE_TIME_SUBMIT_BIT);
	device->initWSIBuffers(cmd);

	m_width = device->getDisplayWidth();
	m_height = device->getDisplayHeight();

	resize(m_width, m_height);

	dc->getDefaultQueue()->flushCommandBuffer(cmdID, NULL);


    m_ready = true;

}

void VulkanAppContext::resize(uint32_t inWidth, uint32_t inHeight){
	VulkanDC *dc = VulkanDC::Get();
	VulkanDC::Device::Queue::CommandBufferID  cmdID = INIT_COMMAND_ID;
	VulkanDC::Device *device = dc->getDefaultDevice();
	device->waitIdle();

	m_renderer->resize(inWidth, inHeight);

	dc->getDefaultQueue()->flushCommandBuffer(cmdID);

}

void VulkanAppContext::render(){
	if (!m_ready) return;

	VulkanDC *dc = VulkanDC::Get();
	VulkanDC::Device *device = dc->getDefaultDevice();

	VkFence nullFence = { VK_NULL_HANDLE };

	m_rot_y += 0.0125;

	if (m_rotor_node){
		m_rotor_node->getNode()->setRotation(0.0, 0.0, -m_rot_y*32.0);
	}

	m_current_time += 0.016;

	if (m_current_time > m_animation.getEndTime()){
		m_current_time = 0.0;
	}


	m_renderer->update();

}

void VulkanAppContext::initAppContext(){

	VkResult error;
	char *extension_names[64];
	char *layer_names[64];

	VkExtensionProperties *instanceExtensions = nullptr;
	VkPhysicalDevice *physicalDevices = nullptr;
	VkLayerProperties *instanceLayers = nullptr;;
	VkLayerProperties *deviceLayers = nullptr;

	uint32_t instanceExtensionCount = 0;
	uint32_t instanceLayerCount = 0;
	uint32_t enabledExtensionCount = 0;
	uint32_t enabledLayerCount = 0;

	/*
		Set up instance layers.
	*/
	char *instanceValidationLayers[] = {
		"MemTracker"
	};

	char *deviceValidationLayers[] = {
		"MemTracker"
	};

	VkBool32 foundValidationLayers = 0;
	error = vkEnumerateInstanceLayerProperties(&instanceLayerCount, NULL);
	assert(!error);

	instanceLayers = (VkLayerProperties*)malloc(sizeof(VkLayerProperties) * instanceLayerCount);
	error = vkEnumerateInstanceLayerProperties(&instanceLayerCount, instanceLayers);
	assert(!error);


	/*
		Set up instance extensions.
	*/

	error = vkEnumerateInstanceExtensionProperties(NULL, &instanceExtensionCount, NULL);
	assert(!error);

	VkBool32 surfaceExtensionFound = 0;
	VkBool32 platformExtensionFound = 0;

	memset(extension_names, 0, sizeof(extension_names));
	instanceExtensions = (VkExtensionProperties*)malloc(sizeof(VkExtensionProperties)*instanceExtensionCount);

	error = vkEnumerateInstanceExtensionProperties(NULL, &instanceExtensionCount, instanceExtensions);
	assert(!error);

	/*
		Look for surface extension.
	*/

	for (uint32_t i = 0; i < instanceExtensionCount; ++i){
		if (!strcmp(VK_KHR_SURFACE_EXTENSION_NAME, instanceExtensions[i].extensionName)){
			surfaceExtensionFound = 1;
			extension_names[enabledExtensionCount++] = VK_KHR_SURFACE_EXTENSION_NAME;
		}

#if defined(WIN32)
		if (!strcmp(VK_KHR_WIN32_SURFACE_EXTENSION_NAME, instanceExtensions[i].extensionName)){
			platformExtensionFound = 1;
			extension_names[enabledExtensionCount++] = VK_KHR_WIN32_SURFACE_EXTENSION_NAME;
		}
#else
	if (!strcmp(VK_KHR_XCB_SURFACE_EXTENSION_NAME, instanceExtensions[i].extensionName)){
			platformExtensionFound = 1;
			extension_names[enabledExtensionCount++] = VK_KHR_XCB_SURFACE_EXTENSION_NAME;
		}
#endif

		assert(enabledExtensionCount < 64);
	}

	if (!surfaceExtensionFound){
#if defined(WIN32)
		MessageBoxA(NULL, "Could not find surface extension.", "ERROR", MB_OK);
#else
		printf("ERROR : Could not find surface extension.\n");
#endif
		exit(1);
	}

	if (!platformExtensionFound){
#if defined(WIN32)
		MessageBoxA(NULL, "Could not find platform surface extension", "Error", MB_OK);
#else
		printf("ERROR : Could not find platform surface extension.\n");
#endif
		exit(1);
	}


	/*
		Set up the application create info.
	*/
	VkApplicationInfo appInfo;
	appInfo.sType = VK_STRUCTURE_TYPE_APPLICATION_INFO;
	appInfo.pNext = NULL;
	appInfo.pApplicationName = "Vulkan Context App\0";
	appInfo.applicationVersion = 1;
	appInfo.pEngineName = "Chris's Engine\0";
	appInfo.engineVersion = 1;
	appInfo.apiVersion = VK_API_VERSION;
	/*
		Set up the instance create info.
	*/
	VkInstanceCreateInfo createInfo;
	createInfo.sType = VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO;
	createInfo.pNext = NULL;
	createInfo.pApplicationInfo = &appInfo;
	createInfo.enabledLayerCount = enabledLayerCount;
	createInfo.ppEnabledLayerNames = (const char * const *)layer_names;
	createInfo.enabledExtensionCount = enabledExtensionCount;
	createInfo.ppEnabledExtensionNames = (const char * const *)extension_names;
	
	/*
		Create the Vulkan Instance
	*/

    VKA_CHECK_ERROR(vulkanCreateInstance(&createInfo, &m_vk_instance), "Could not create Vulkan Instance");

	/*
		Set up and initialise the Vulkan Device Context.
	*/
	VulkanDC *dc = VulkanDC::Get();
	dc->initDC(m_vk_instance);

	fpGetPhysicalDeviceSurfaceCapabilitiesKHR = (PFN_vkGetPhysicalDeviceSurfaceCapabilitiesKHR)vkGetInstanceProcAddr(m_vk_instance, "vkGetPhysicalDeviceSurfaceCapabilitiesKHR");
	fpGetPhysicalDeviceSurfaceFormatsKHR = (PFN_vkGetPhysicalDeviceSurfaceFormatsKHR)vkGetInstanceProcAddr(m_vk_instance, "vkGetPhysicalDeviceSurfaceFormatsKHR");
	fpGetPhysicalDeviceSurfacePresentModesKHR = (PFN_vkGetPhysicalDeviceSurfacePresentModesKHR)vkGetInstanceProcAddr(m_vk_instance, "vkGetPhysicalDeviceSurfacePresentModesKHR");
	fpGetPhysicalDeviceSurfaceSupportKHR = (PFN_vkGetPhysicalDeviceSurfaceSupportKHR)vkGetInstanceProcAddr(m_vk_instance, "vkGetPhysicalDeviceSurfaceSupportKHR");
	fpCreateSwapchainKHR = (PFN_vkCreateSwapchainKHR)vkGetInstanceProcAddr(m_vk_instance, "vkCreateSwapchainKHR");
	fpDestroySwapchainKHR = (PFN_vkDestroySwapchainKHR)vkGetInstanceProcAddr(m_vk_instance, "vkDestroySwapchainKHR");
	fpGetSwapchainImagesKHR = (PFN_vkGetSwapchainImagesKHR)vkGetInstanceProcAddr(m_vk_instance, "vkGetSwapchainImagesKHR");
	fpAcquireNextImageKHR = (PFN_vkAcquireNextImageKHR)vkGetInstanceProcAddr(m_vk_instance, "vkAcquireNextImageKHR");
	fpQueuePresentKHR = (PFN_vkQueuePresentKHR)vkGetInstanceProcAddr(m_vk_instance, "vkQueuePresentKHR");



}


void VulkanAppContext::initWSI(xcb_connection_t *inConnection, xcb_screen_t *inScreen, xcb_drawable_t inWindow){

	VulkanDC *dc = VulkanDC::Get();
	VulkanDC::Device *device = dc->getDefaultDevice();
	if(!device) return;

	device->initWSI(inConnection,inScreen,inWindow);

}


void VulkanAppContext::initQueue(){

	/*
		Get the default graphics queue.
	*/
	VulkanDC *dc = VulkanDC::Get();
    VulkanDC::Device::Queue::Name queueName = "DEFAULT_GRAPHICS_QUEUE";
    VulkanDC::Device::Queue *queue = dc->getQueueForGraphics(queueName,m_surface_format);
    dc->setDefaultDevice(queue->getDevice());
	dc->setDefaultQueue(queue);

}

VulkanAppContext::~VulkanAppContext()
{
}

void VulkanAppContext::shutDown(){

	VulkanDC *dc = VulkanDC::Get();
	VulkanDC::Device *device = dc->getDefaultDevice();
	VulkanDC::Device::Queue *queue = dc->getDefaultQueue();

	device->waitIdle();

	m_renderer->shutDown();



}

bool VulkanAppContext::initPrograms(){


	/*
		Check that the programs are valid.
	*/
	return true;
}

VkeMaterial *VulkanAppContext::getMaterial(VkeMaterial::ID inID){
	return m_materials.getMaterial(inID);

}

void VulkanAppContext::setCameraMatrix(nv_math::mat4f &inMat){
	((RENDERER*)m_renderer)->setCameraLookAt(inMat);

}
