/*-----------------------------------------------------------------------
Copyright (c) 2014-2016, NVIDIA. All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
* Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.
* Neither the name of its contributors may be used to endorse
or promote products derived from this software without specific
prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
-----------------------------------------------------------------------*/
/* Contact chebert@nvidia.com (Chris Hebert) for feedback */



#include<iostream>

#include<xcb/xcb.h>
#include<unistd.h>
#include<stdio.h>
#define VK_USE_PLATFORM_XCB_KHR

#include<stdint.h>
#include<assert.h>

#include<vulkan/vulkan.h>
#include<vulkan/vk_platform.h>
#include"nv_math/nv_math.h"
#include"cameracontrol.hpp"

#include"VulkanAppContext.h"

#define TUTORIAL_WIDTH 1280
#define TUTORIAL_HEIGHT 800



struct Rct{
	uint32_t width;
	uint32_t height;
};

struct Pnt{
	Pnt():x(0),y(0){}
	Pnt(const uint32_t inX, const uint32_t inY):x(inX),y(inY){}
	~Pnt(){};

	uint32_t x;
	uint32_t y;
};

struct MouseInfo{
	struct Pnt clickPos;
	struct Pnt mousePos;
	bool mouseState;
	uint32_t mouseFlags;
};

struct SampleInstance{
	bool running;


	xcb_connection_t *xcbconn;
	xcb_screen_t *xcbscreen;
	xcb_drawable_t xcbwin;

	xcb_intern_atom_cookie_t wmProtocolsCookie;
	xcb_intern_atom_reply_t *wmProtocolsReply;

	xcb_intern_atom_cookie_t wmDeleteCookie;
	xcb_intern_atom_reply_t *wmDeleteReply;

	nv_helpers::CameraControl control;

	char *name;

	struct MouseInfo mouseInfo;
	struct Rct rct;
	VkAllocationCallbacks callbacks;
	VkInstance vulkanInstance;

	nv_math::vec4f viewport;

	float rotation;
	bool ready;



}sample;


/*******************************************/
/*Alloc Callbacks ****************************/
/*Vulkan uses these methods to  *************/
/*Allocate and free memory *****************/
/********************************************/

VKAPI_ATTR void * VKAPI_CALL customAllocFunc(void *userData, size_t size, size_t alignment, VkSystemAllocationScope allocationScope){
	return malloc(size);
}


VKAPI_ATTR void *VKAPI_CALL customReallocFunc(void *userData, void *originalData, size_t size, size_t allignment, VkSystemAllocationScope allocationScope){
	return realloc(originalData, size);
}

VKAPI_ATTR void VKAPI_CALL customFreeFunc(void *userData, void *mem){
	free(mem);
}


/********************************************/
/*Vulkan Initialization ************************/
/********************************************/

void initVulkan(){

	VulkanAppContext *ctxt = VulkanAppContext::GetInstance();
	ctxt->initAppContext();


}



void renderScene(){

	VulkanAppContext *ctxt = VulkanAppContext::GetInstance();

	nv_math::vec2i screenSize(sample.rct.width,sample.rct.height);
	nv_math::vec2f mousePos((float)sample.mouseInfo.mousePos.x,(float)sample.mouseInfo.mousePos.y);

	sample.control.processActions(screenSize,mousePos,sample.mouseInfo.mouseFlags,0);
	ctxt->setCameraMatrix(sample.control.m_viewMatrix);

	ctxt->render();
}


void xLibMessagePump(){

	xcb_generic_event_t *xcbevt;

	static bool isDone = false;
	bool isReady = false;

	while(!isDone){
		xcb_flush(sample.xcbconn);

		while((xcbevt = xcb_poll_for_event(sample.xcbconn))){

			switch(xcbevt->response_type & ~0x80){
				case XCB_EXPOSE:{
							isReady = true;
				}break;

				/********************************
				* Key Button Press Event
				********************************/
				case XCB_BUTTON_PRESS:{
						xcb_button_press_event_t *bp = (xcb_button_press_event_t *) xcbevt;

						switch(bp->detail){
						/*
						 * Mouse wheel up
						 **/
						case 4:


							break;

							/*
							 * Mouse wheel down
							 */
						case 5:


							break;
							/*
							 * Any other button
							 */
						default:

							sample.mouseInfo.mousePos.x = bp->event_x;
							sample.mouseInfo.mousePos.y = bp->event_y;
							sample.mouseInfo.clickPos.x = bp->event_x;
							sample.mouseInfo.clickPos.y = bp->event_y;
							sample.mouseInfo.mouseState = true;
							sample.mouseInfo.mouseFlags |= 0x1;

							break;
						}


				}	break;

				/********************************
				* Key Button Release Event
				********************************/
				case XCB_BUTTON_RELEASE:{

					xcb_button_release_event_t *br = (xcb_button_release_event_t*) xcbevt;

					sample.mouseInfo.mousePos.x = br->event_x;
					sample.mouseInfo.mousePos.y = br->event_y;
					sample.mouseInfo.mouseState = false;
					sample.mouseInfo.mouseFlags = 0x0;


				}
				break;

				/********************************
				* Key Motion Event
				********************************/
				case XCB_MOTION_NOTIFY:{
						xcb_motion_notify_event_t *motion = (xcb_motion_notify_event_t*)xcbevt;
						sample.mouseInfo.mousePos.x = motion->event_x;
						sample.mouseInfo.mousePos.y = motion->event_y;



				}break;

				/********************************
				* Key Enter  Window Event
				********************************/
				case XCB_ENTER_NOTIFY:{

					xcb_enter_notify_event_t *enter = (xcb_enter_notify_event_t *) xcbevt;
					sample.mouseInfo.mouseState = false;
					sample.mouseInfo.mouseFlags = 0x0;



				}break;

				/********************************
				 * Key Leave Window Event
				 ********************************/
				case XCB_LEAVE_NOTIFY:{
						xcb_leave_notify_event_t *leave = (xcb_leave_notify_event_t *) xcbevt;
						sample.mouseInfo.mouseState = false;
						sample.mouseInfo.mouseFlags = 0x0;

				}break;

				/********************************
				 * Key Press Event
				 ********************************/
				case XCB_KEY_PRESS:{
						xcb_key_press_event_t *kp = (xcb_key_press_event_t *) xcbevt;


				}break;

				/********************************
				 * Key Release Event
				 ********************************/
				case XCB_KEY_RELEASE:{
					xcb_key_release_event_t *kr = (xcb_key_release_event_t *) xcbevt;



				}break;

				/*
				 *  Window closed. Kill the app.
				 */

				case XCB_CLIENT_MESSAGE:{

					if((*(xcb_client_message_event_t*)xcbevt).data.data32[0] == (*sample.wmDeleteReply).atom){

						isDone = true;
					}

				}break;
				default:

				break;
			}
		free(xcbevt);
		}

		if(isDone){
			//clean up the application.

			VulkanAppContext *ctxt = VulkanAppContext::GetInstance();
			ctxt->shutDown();


		}else{
			if(isReady){
				renderScene();
			}
		}

	}

}


void createTutorialWindow(){


	const xcb_setup_t *setup;
	xcb_screen_iterator_t itr;
	int scr;

	sample.xcbconn 	=	xcb_connect(NULL,&scr);

	setup = xcb_get_setup(sample.xcbconn);

	itr 	=	xcb_setup_roots_iterator(setup);
	while(scr-- > 0) xcb_screen_next(&itr);

	sample.xcbscreen = itr.data;

	sample.xcbwin		=	xcb_generate_id(sample.xcbconn);

	uint32_t mask = 0;
	uint32_t values[1];

	mask = XCB_CW_EVENT_MASK;
	values[0] = XCB_EVENT_MASK_EXPOSURE | XCB_EVENT_MASK_BUTTON_PRESS |
			XCB_EVENT_MASK_BUTTON_RELEASE | XCB_EVENT_MASK_POINTER_MOTION |
			XCB_EVENT_MASK_ENTER_WINDOW | XCB_EVENT_MASK_LEAVE_WINDOW |
			XCB_EVENT_MASK_KEY_PRESS | XCB_EVENT_MASK_KEY_RELEASE;;

	xcb_create_window(sample.xcbconn,
		XCB_COPY_FROM_PARENT,
		sample.xcbwin,
		sample.xcbscreen->root,
		0,0,sample.rct.width,sample.rct.height,
		10,
		XCB_WINDOW_CLASS_INPUT_OUTPUT,
		sample.xcbscreen->root_visual,
		mask,values);

	sample.wmProtocolsCookie = xcb_intern_atom(sample.xcbconn,1,12,"WM_PROTOCOLS");
	sample.wmProtocolsReply= xcb_intern_atom_reply(sample.xcbconn,sample.wmProtocolsCookie,0);

	sample.wmDeleteCookie= xcb_intern_atom(sample.xcbconn,0,16,"WM_DELETE_WINDOW");
	sample.wmDeleteReply = xcb_intern_atom_reply(sample.xcbconn,sample.wmDeleteCookie,0);

	xcb_change_property(sample.xcbconn,XCB_PROP_MODE_REPLACE,sample.xcbwin,(*sample.wmProtocolsReply).atom,4,32,1,&(*sample.wmDeleteReply).atom);


	//xcb_flush(tutorial.xcbconn);
	xcb_map_window(sample.xcbconn,sample.xcbwin);
	const uint32_t crds[] = {100,100};
	xcb_configure_window(sample.xcbconn,sample.xcbwin,XCB_CONFIG_WINDOW_X | XCB_CONFIG_WINDOW_Y,crds);

//	xcb_flush(sample.xcbconn);






}

/********************************************/
/*Win32 Win Main ****************************/
/********************************************/

int main(int argc, char **argv){



	sample.name = "Chopper";
	sample.rct.width = TUTORIAL_WIDTH;
	sample.rct.height = TUTORIAL_HEIGHT;
	sample.running = true;
	sample.ready = false;

	initVulkan();

	sample.control.m_sceneOrbit = nv_math::vec3f(0.0);
	sample.control.m_sceneDimension = float(1.0);
	nv_math::vec3f scenePos = nv_math::vec3f(20.0,-1.0,8.0);

	sample.control.m_viewMatrix = nv_math::look_at(sample.control.m_sceneOrbit - (scenePos * sample.control.m_sceneDimension * 0.5f), sample.control.m_sceneOrbit,nv_math::vec3f(0,1,0));

	createTutorialWindow();

	VulkanAppContext *ctxt = VulkanAppContext::GetInstance();

	VulkanDC *dc = VulkanDC::Get();
	VulkanDC::Device *device = dc->getDefaultDevice();

	if(!device){
		std::cout << "Device Is NULL" << std::endl;
		exit(1);
	}

	device->setDisplayDimensions(sample.rct.width,sample.rct.height);
	device->initWSI(sample.xcbconn,sample.xcbscreen,sample.xcbwin);
	ctxt->initQueue();

	ctxt->initRenderer();
	xLibMessagePump();
	return 0;
}

