
#include "VulkanDeviceContext.h"
#include "VkeCreateUtils.h"
#include"vkaUtils.h"
#include"VulkanAppContext.h"
#include<iostream>
VulkanDC::Device::Queue::Queue(){

}

VulkanDC::Device::Queue::Queue(VulkanDC::Device::Queue::Name &inName, VulkanDC::Device::Queue::NodeID inNodeID) :
m_name(inName),
m_node_id(inNodeID),
m_queue(VK_NULL_HANDLE){}

VulkanDC::Device::Queue::Queue(const VulkanDC::Device::Queue &inOther){
	m_name = inOther.m_name;
	m_node_id = inOther.m_node_id;
	m_queue = inOther.m_queue;
}

void VulkanDC::Device::Queue::initQueue(VulkanDC::Device *inDevice){
	//TODO - currently this is just queue index 0.
	//need to work out how best to use this.
	m_device = inDevice;
	VkDevice device = m_device->getVKDevice();
	vkGetDeviceQueue(device, m_node_id, 0, &m_queue);

	VkCommandPoolCreateInfo cmdPoolInfo;
	commandPoolCreateInfo(&cmdPoolInfo, m_node_id, VK_COMMAND_POOL_CREATE_RESET_COMMAND_BUFFER_BIT);
	VKA_CHECK_ERROR(vkCreateCommandPool(device,&cmdPoolInfo,NULL,&m_command_pool),"Could not create command Pool.\n");
}



void VulkanDC::Device::Queue::createCommandBuffer(
	VkCommandBuffer *outBuffer,
	VkCommandBufferLevel inLevel){

	VkCommandBufferAllocateInfo cmd;
	memset(&cmd, 0, sizeof(cmd));
	cmd.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
	cmd.commandPool= m_command_pool;
	cmd.level = inLevel;
	cmd.commandBufferCount = 1;

	vkAllocateCommandBuffers(m_device->getVKDevice(), &cmd, outBuffer);
}

bool VulkanDC::Device::Queue::createAndCacheCommandBuffer(
	const VulkanDC::Device::Queue::CommandBufferID &inID,
	VkCommandBuffer *outBuffer,
	VkCommandBufferLevel inLevel
	){

	bool isNew = true;

	CommandBufferMap::iterator itr = m_cached_buffers.find(inID);

	if (itr == m_cached_buffers.end() || itr->second == VK_NULL_HANDLE){
		createCommandBuffer(outBuffer, inLevel);
		m_cached_buffers[inID] = *outBuffer;
	}
	else{
		*outBuffer = itr->second;
		isNew = false;
	}
	return isNew;
}

void VulkanDC::Device::Queue::beginCommandBuffer(
	VkCommandBuffer *inBuffer,
	VkCommandBufferUsageFlags inFlags,
	VkCommandBufferLevel inLevel,
	VkFramebuffer inFBO,
	VkRenderPass inRP
	){

	if (*inBuffer == VK_NULL_HANDLE){
		createCommandBuffer(inBuffer, inLevel);
	}

	VkCommandBufferBeginInfo cmdBufInfo;
	memset(&cmdBufInfo, 0, sizeof(cmdBufInfo));

	cmdBufInfo.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
	cmdBufInfo.flags = inFlags;

	VKA_CHECK_ERROR(vkBeginCommandBuffer(*inBuffer, &cmdBufInfo), "Could not begin command buffer.\n");

}

void VulkanDC::Device::Queue::beginCommandBuffer(
	VulkanDC::Device::Queue::CommandBufferID &inID,
	VkCommandBuffer *outBuffer,
	VkCommandBufferUsageFlags inFlags,
	VkCommandBufferLevel inLevel,
	VkFramebuffer inFBO,
	VkRenderPass inRP
	){

	if (!createAndCacheCommandBuffer(inID, outBuffer, inLevel)) return;


	VkCommandBufferBeginInfo cmdBufInfo;
	memset(&cmdBufInfo, 0, sizeof(cmdBufInfo));

	cmdBufInfo.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
	cmdBufInfo.flags = inFlags;

	VKA_CHECK_ERROR(vkBeginCommandBuffer(*outBuffer, &cmdBufInfo), "Could not begin command buffer.\n");

}

VkCommandBuffer VulkanDC::Device::Queue::getCachedCommandBuffer(VulkanDC::Device::Queue::CommandBufferID &inID){
	VulkanDC::Device::Queue::CommandBufferMap::iterator itr = m_cached_buffers.find(inID);
	if (itr == m_cached_buffers.end()) return VK_NULL_HANDLE;

	return itr->second;
}

void VulkanDC::Device::Queue::resetCommandBuffer(VulkanDC::Device::Queue::CommandBufferID &inID){
	VkCommandBuffer cmd = getCachedCommandBuffer(inID);
	if (cmd == VK_NULL_HANDLE) return;

	VkResult rslt = vkResetCommandBuffer(cmd, 0);

	VkCommandBufferBeginInfo cmdBufInfo;
	memset(&cmdBufInfo, 0, sizeof(cmdBufInfo));

	cmdBufInfo.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
	cmdBufInfo.flags = 0;

	VKA_CHECK_ERROR(vkBeginCommandBuffer(cmd, &cmdBufInfo), "Could not begin command buffer.\n");

}

void VulkanDC::Device::Queue::flushPersistentBuffer(VulkanDC::Device::Queue::CommandBufferID &inID, VkFence *inFence){

	VkCommandBuffer cmd = getCachedCommandBuffer(inID);
	if (cmd == VK_NULL_HANDLE) return;

	VKA_CHECK_ERROR(vkEndCommandBuffer(cmd), "Could not end command buffer.\n");
	VkCommandBuffer bufs[] = { cmd };
	VkFence theFence = VK_NULL_HANDLE;
	if (inFence){
		theFence = *inFence;
	}

	VkSubmitInfo subInfo;
	memset(&subInfo, 0, sizeof(subInfo));
	subInfo.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
	subInfo.commandBufferCount = 1;
	subInfo.pCommandBuffers = bufs;

	VKA_CHECK_ERROR(vkQueueSubmit(m_queue, 1, &subInfo, theFence), "Could not submit queue.\n");

}

void VulkanDC::Device::Queue::flushCommandBuffer(VulkanDC::Device::Queue::CommandBufferID &inID, VkFence *inFence, bool inDestroy){

	VkCommandBuffer cmd = getCachedCommandBuffer(inID);
	if (cmd == VK_NULL_HANDLE) return;

	VKA_CHECK_ERROR(vkEndCommandBuffer(cmd), "Could not end command buffer.\n");

	VkCommandBuffer bufs[] = { cmd };
	VkFence theFence = VK_NULL_HANDLE;
	bool createdFence = false;
	if (inFence){
		theFence = *inFence;
	}
	else if(inDestroy){
		VkFenceCreateInfo fenceInfo = { VK_STRUCTURE_TYPE_FENCE_CREATE_INFO };
		vkCreateFence(m_device->getVKDevice(), &fenceInfo, NULL, &theFence);
		createdFence = true;
	}

	VkSubmitInfo subInfo;
	memset(&subInfo, 0, sizeof(subInfo));
	subInfo.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
	subInfo.commandBufferCount = 1;
	subInfo.pCommandBuffers = bufs;

	VKA_CHECK_ERROR(vkQueueSubmit(m_queue, 1, &subInfo, theFence), "Could not submit queue.\n");

	if (inDestroy){
		VKA_CHECK_ERROR(vkWaitForFences(m_device->getVKDevice(), 1, &theFence, VK_TRUE, 100000000000), "Could not wait for idle queue.\n");

		destroyCachedCommandBuffer(inID);

		if (createdFence){
			vkDestroyFence(m_device->getVKDevice(), theFence, NULL);
		}
	}
}

void VulkanDC::Device::Queue::destroyCachedCommandBuffer(VulkanDC::Device::Queue::CommandBufferID &inID){
	VkCommandBuffer cmd = getCachedCommandBuffer(inID);
	if (cmd == VK_NULL_HANDLE) return;
	vkFreeCommandBuffers(m_device->getVKDevice(), m_command_pool, 1, &cmd);

	m_cached_buffers[inID] = VK_NULL_HANDLE;

}

VulkanDC::Device::Queue::~Queue(){}


VulkanDC::Device::Device(){}
VulkanDC::Device::Device(const VkPhysicalDevice &inDevice) :
m_physical_device(inDevice),
m_device(VK_NULL_HANDLE),
m_swap_chain_buffers(0),
m_swapchain(VK_NULL_HANDLE){
	initDevice();
}

VulkanDC::Device::Queue *VulkanDC::Device::getQueue(VulkanDC::Device::Queue::Name &inName){
	VulkanDC::Device::Queue::Map::iterator itr;
	itr = m_queues.find(inName);

	if (itr == m_queues.end()) return NULL;
	return itr->second;
}

uint32_t VulkanDC::Device::getBestMemoryType(uint32_t inType, VkFlags inFlags){

	for (uint32_t i = 0; i < 32; ++i){

		if ((inType & 1) == 1){
			if ((m_memory_properties.memoryTypes[i].propertyFlags & inFlags) == inFlags){
				return i;
			}
		}
		inType >>= 1;
	}

	return 0;
}

void VulkanDC::Device::initWSIBuffers(VkCommandBuffer &inInitCommand){

	VkResult error;
	VulkanAppContext *ctxt = VulkanAppContext::GetInstance();

	VkSwapchainKHR oldSwapChain = m_swapchain;

	VkSurfaceCapabilitiesKHR surfaceCaps;
	error = ctxt->fpGetPhysicalDeviceSurfaceCapabilitiesKHR(m_physical_device, m_surface, &surfaceCaps);
	assert(!error);

	uint32_t presentModeCount;

	//Get the count of available present modes.
	error = ctxt->fpGetPhysicalDeviceSurfacePresentModesKHR(m_physical_device, m_surface, &presentModeCount, NULL);
	assert(!error);

	//allocate storage for the present modes
 	VkPresentModeKHR *presentModes = (VkPresentModeKHR*)malloc(sizeof(VkPresentModeKHR)*presentModeCount);
	assert(presentModes);

	//Get the actual present modes, returned to the storage we just allocated.
	error = ctxt->fpGetPhysicalDeviceSurfacePresentModesKHR(m_physical_device, m_surface, &presentModeCount, presentModes);
	assert(!error);

	VkExtent2D swapChainExtent;

	if (surfaceCaps.currentExtent.width == -1){
		/*
			if this is the first time we've created
			the swap chain (ie we have not resized)
			then we wont find a width or height, so we
			will set the extent to our desired width
			and height.
		*/
		swapChainExtent.width = m_width;
		swapChainExtent.height = m_height;
	}
	else{
		swapChainExtent = surfaceCaps.currentExtent;
		m_width = surfaceCaps.currentExtent.width;
		m_height = surfaceCaps.currentExtent.height;
	}
	VkPresentModeKHR swapChainPresentMode = VK_PRESENT_MODE_FIFO_KHR;

	uint32_t requiredSwapChainImages = surfaceCaps.minImageCount + 1;
	if ((surfaceCaps.maxImageCount > 0) && (requiredSwapChainImages > surfaceCaps.maxImageArrayLayers)){
		requiredSwapChainImages = surfaceCaps.maxImageCount;
	}

	VkSurfaceTransformFlagBitsKHR preTransform;
	if (surfaceCaps.supportedTransforms & VK_SURFACE_TRANSFORM_IDENTITY_BIT_KHR){
		preTransform = VK_SURFACE_TRANSFORM_IDENTITY_BIT_KHR;
	}
	else{
		preTransform = surfaceCaps.currentTransform;
	}


	/*
		We configure the swap chain with a create info
		structure as usual with:

		surface				-		The surface to present to.
		minImageCount		-		The minimum number of images for our swap chain
		imageFormat			-		The color format for the present.
		imageColorSpace		-		The color space for the present.
		extent				-		The width and height of the swap chain image.
		preTransform		-		The transformation (eg rotate 90 ccw) to apply before presenting.
		imageArrayLayers	-		The number of layers in the image
		imageSharingMode	-		It is possible to share swapchains between queues, in our case we only have 1 queue.
		presentMode			-		The present mode (see above) used by this swap chain.
		oldSwapChain		-		if this swap chain is being recreated because of a resize, some resources may be reused.
		clipped				-		Determines whether or not every pixel is owned by Vulkan (if eg a window obscures our Vulkan window)

	*/
	uint32_t queueIdx = 0;

	VkSwapchainCreateInfoKHR swapChainInfo = { VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR };
	swapChainInfo.surface = m_surface;
	swapChainInfo.minImageCount = requiredSwapChainImages;
	swapChainInfo.imageFormat = m_format;
	swapChainInfo.imageColorSpace = m_color_space;
	swapChainInfo.imageExtent.width = swapChainExtent.width;
	swapChainInfo.imageExtent.height = swapChainExtent.height;
	swapChainInfo.imageUsage = VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT;
	swapChainInfo.preTransform = preTransform;
	swapChainInfo.imageArrayLayers = 1;
	swapChainInfo.imageSharingMode = VK_SHARING_MODE_EXCLUSIVE;
	swapChainInfo.queueFamilyIndexCount = 1;
	swapChainInfo.pQueueFamilyIndices = &queueIdx;
	swapChainInfo.presentMode = swapChainPresentMode;
	swapChainInfo.oldSwapchain = oldSwapChain;
	swapChainInfo.clipped = true;


	uint32_t i;

	error = ctxt->fpCreateSwapchainKHR(m_device, &swapChainInfo, NULL, &m_swapchain);
	assert(!error);

	/*
		Destroy the old swap chain if one exists.
	*/
	if (oldSwapChain != VK_NULL_HANDLE){
		ctxt->fpDestroySwapchainKHR(m_device, oldSwapChain, NULL);
	}

	error = ctxt->fpGetSwapchainImagesKHR(m_device, m_swapchain, &m_swapchain_image_count, NULL);

	//Allocate storage for the swap chain images.
	VkImage *swapChainImages = (VkImage*)malloc(m_swapchain_image_count * sizeof(VkImage));
	assert(swapChainImages);

	//Get the swapchain images.
	error = ctxt->fpGetSwapchainImagesKHR(m_device, m_swapchain, &m_swapchain_image_count, swapChainImages);
	assert(!error);

	m_swap_chain_buffers = (SwapchainBuffers*)malloc(sizeof(SwapchainBuffers) * m_swapchain_image_count);
	assert(m_swap_chain_buffers);


	for (i = 0; i < m_swapchain_image_count; ++i){

		VkImageViewCreateInfo colorAttachmentView = { VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO };
		colorAttachmentView.format = m_format;
		colorAttachmentView.components.r = VK_COMPONENT_SWIZZLE_R;
		colorAttachmentView.components.g = VK_COMPONENT_SWIZZLE_G;
		colorAttachmentView.components.b = VK_COMPONENT_SWIZZLE_B;
		colorAttachmentView.components.a = VK_COMPONENT_SWIZZLE_A;
		colorAttachmentView.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
		colorAttachmentView.subresourceRange.baseArrayLayer = 0;
		colorAttachmentView.subresourceRange.baseMipLevel = 0;
		colorAttachmentView.subresourceRange.layerCount = 1;
		colorAttachmentView.subresourceRange.levelCount = 1;
		colorAttachmentView.viewType = VK_IMAGE_VIEW_TYPE_2D;

		/*
			We then populate our SwapChainBuffers structure for
			each image.
		*/
		m_swap_chain_buffers[i].image = swapChainImages[i];

		/*
			When an image is first created, it is created with the
			image layout VK_IMAGE_LAYOUT_UNDEFINED. This layout
			is not usable for presenting so needs to be transitioned
			away to a usable format (VK_IMAGE_LAYOUT_PRESENT_SRC_KHR).

			The image layout transition happens (in this case) as part of
			a pipeline barrier (see method definition for setImageLayout)
		*/


		imageSetLayout(&inInitCommand,m_swap_chain_buffers[i].image, VK_IMAGE_ASPECT_COLOR_BIT, VK_IMAGE_LAYOUT_UNDEFINED, VK_IMAGE_LAYOUT_PRESENT_SRC_KHR);

		colorAttachmentView.image = m_swap_chain_buffers[i].image;
		m_swap_chain_buffers[i].cmd = VK_NULL_HANDLE;

		/*
			Create the image view.
		*/
		error = vkCreateImageView(m_device, &colorAttachmentView, NULL, &m_swap_chain_buffers[i].view);

		assert(!error);
	}

	m_current_swap_chain_buffer = 0;

}

void VulkanDC::Device::initWSI(xcb_connection_t *inConnection, xcb_screen_t *inScreen, xcb_drawable_t inWindow){




	VkResult error;
	VulkanAppContext *ctxt = VulkanAppContext::GetInstance();

	m_xcbconn = inConnection;
	m_xcbscreen = inScreen;
	m_xcbwin = inWindow;

	VkXcbSurfaceCreateInfoKHR surfInfo = { VK_STRUCTURE_TYPE_XCB_SURFACE_CREATE_INFO_KHR };
	surfInfo.connection = inConnection;
	surfInfo.window = inWindow;
	surfInfo.flags = 0;
	error = vkCreateXcbSurfaceKHR(ctxt->getVKInstance(),&surfInfo,NULL, &m_surface);


	VkBool32 *supportsPresent = (VkBool32*)malloc(sizeof(VkBool32) * m_queue_count);
	for (uint32_t i = 0; i < m_queue_count; ++i){
		ctxt->fpGetPhysicalDeviceSurfaceSupportKHR(m_physical_device, i, m_surface, &supportsPresent[i]);
	}

	uint32_t graphicsQueueNodeIndex = UINT32_MAX;
	uint32_t presentQueueNodeIndex = UINT32_MAX;

	for (uint32_t i = 0; i < m_queue_count; ++i){

		/*
			and look for the VK_QUEUE_GRAPHICS_BIT.
		*/
		if ((m_queue_properties[i].queueFlags & VK_QUEUE_GRAPHICS_BIT) != 0){
			if (graphicsQueueNodeIndex == UINT32_MAX){
				/*
					If we haven't already found a queue for graphics,
					we'll try this one.
				*/
				graphicsQueueNodeIndex = i;
			}

			if (supportsPresent[i] == VK_TRUE){
				/*
					If our array of booleans tells us
					that this particular queue also supports
					presetation, then we are done and we
					can exit the loop.
				*/
				graphicsQueueNodeIndex = i;
				presentQueueNodeIndex = i;
				break;
			}
		}
	}


	if (presentQueueNodeIndex == UINT32_MAX){
		for (uint32_t i = 0; i < m_queue_count; ++i){
			if (supportsPresent[i] == VK_TRUE){
				presentQueueNodeIndex = i;
				break;
			}
		}
	}
	free(supportsPresent);

	if (graphicsQueueNodeIndex == UINT32_MAX || presentQueueNodeIndex == UINT32_MAX){
#if defined(WIN32)
		MessageBoxA(NULL, "Could not find suitable queue.", "Error", MB_OK);
#else
		printf("Could not find suitable queue.\n");
#endif
		exit(1);
	}

	/*
		Since on our hardware all queues should support graphics and
		present, if there is a difference in the two indices that we
		retrieved, then something went wrong and we should exit.
	*/
	if (graphicsQueueNodeIndex != presentQueueNodeIndex){
#if defined(WIN32)
		MessageBoxA(NULL, "Could not find a common graphics and present queue", "Error", MB_OK);
#else
		printf("Could not find a common graphics and present queue.\n");
#endif
		exit(1);
	}

	m_graphics_queue_index =  graphicsQueueNodeIndex;

	uint32_t formatCount;

	error = ctxt->fpGetPhysicalDeviceSurfaceFormatsKHR(m_physical_device, m_surface, &formatCount, NULL);
	assert(!error);

	VkSurfaceFormatKHR *formats = (VkSurfaceFormatKHR *)malloc(sizeof(VkSurfaceFormatKHR)* formatCount);

	error = ctxt->fpGetPhysicalDeviceSurfaceFormatsKHR(m_physical_device, m_surface, &formatCount, formats);
	assert(!error);

	/*
		if only 1 format was found and it is undefined,
		force the format for the tutorial to :

		VK_FORMAT_B8G8R8A8_UNORM.
	*/

	if (formatCount == 1 && formats[0].format == VK_FORMAT_UNDEFINED){
		m_format = VK_FORMAT_B8G8R8A8_UNORM;
	}
	else{
		assert(formatCount > 0);
		/*
			Otherwise, just pick the first format.
		*/
		m_format = formats[0].format;
	}
	m_color_space = formats[0].colorSpace;



}

uint32_t VulkanDC::Device::present(VulkanDC::Device::Queue *inQueue, VkCommandBuffer *inBuffers){


	m_current_swap_chain_buffer = 0;

	VkSemaphore presentCompleteSemaphore;
	VkResult error;
	VulkanAppContext *ctxt = VulkanAppContext::GetInstance();

	VkSemaphoreCreateInfo semInfo = { VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO };

	error = vkCreateSemaphore(m_device, &semInfo, NULL, &presentCompleteSemaphore);
	assert(!error);

	error = ctxt->fpAcquireNextImageKHR(m_device, m_swapchain, UINT64_MAX, presentCompleteSemaphore, NULL, &m_current_swap_chain_buffer);


	if (error == VK_ERROR_OUT_OF_DATE_KHR){
		/*
			Handle Resize and recreate/redraw
		*/
	}
	else{
		assert(!error);
	}

	VkFence nullFence = VK_NULL_HANDLE;
	VkSubmitInfo subInfo = { VK_STRUCTURE_TYPE_SUBMIT_INFO };
	subInfo.waitSemaphoreCount = 1;
	subInfo.pWaitSemaphores = &presentCompleteSemaphore;
	subInfo.commandBufferCount = 1;
	subInfo.pCommandBuffers = &inBuffers[m_current_swap_chain_buffer];//&m_swap_chain_buffers[m_current_swap_chain_buffer].cmd;
	subInfo.signalSemaphoreCount = 0;
	subInfo.pSignalSemaphores = NULL;

	error = vkQueueSubmit(inQueue->getVKQueue(), 1, &subInfo, nullFence);
	assert(!error);

	VkPresentInfoKHR present;
	memset(&present, 0, sizeof(present));
	present.swapchainCount = 1;
	present.pSwapchains = &m_swapchain;
	present.pImageIndices = &m_current_swap_chain_buffer;

	error = ctxt->fpQueuePresentKHR(inQueue->getVKQueue(), &present);

	if (error == VK_ERROR_OUT_OF_DATE_KHR){
		//Resize/redraw.
	}
	else{
		assert(!error);
	}

	error = vkQueueWaitIdle(inQueue->getVKQueue());
	assert(!error);

	vkDestroySemaphore(m_device, presentCompleteSemaphore, NULL);

	return m_current_swap_chain_buffer;

}

void VulkanDC::Device::waitIdle(){
	VKA_CHECK_ERROR(vkDeviceWaitIdle(m_device), "Could not wait for device idle.\n");

}

void VulkanDC::Device::initDevice(){

	char *extension_names[64];
	char *layer_names[64];
	VkResult error;
	uint32_t enabledExtensionCount = 0;
	uint32_t enabledLayerCount = 0;
	VkLayerProperties *layerProps;
	uint32_t propCount;
	VkResult err =	vkEnumerateDeviceLayerProperties(m_physical_device, &propCount, NULL);
	if (err == VK_SUCCESS){
		layerProps = (VkLayerProperties*)malloc(sizeof(VkLayerProperties) * propCount);
		err = vkEnumerateDeviceLayerProperties(m_physical_device, &propCount, layerProps);
	}

	uint32_t deviceExtensionCount = 0;
	VkExtensionProperties *deviceExtensions = NULL;

	//First call, we pass in NULL for the deviceExtensions ptr.
	error = vkEnumerateDeviceExtensionProperties(m_physical_device, NULL, &deviceExtensionCount, NULL);
	assert(!error);

	//We now now how much storage we'll need for the device extensions.
	deviceExtensions = (VkExtensionProperties*)malloc(sizeof(VkExtensionProperties)*deviceExtensionCount);

	//Second call, this time we pass the pointer to the memory we just allocated.
	error = vkEnumerateDeviceExtensionProperties(m_physical_device, NULL, &deviceExtensionCount, deviceExtensions);
	assert(!error);


	/*
		Now that we've retrieved an enumeration of the extensions
		on the chosen physical device, we can search for the
		extension that we need to display.
	*/

	VkBool32 swapChainFound = 0;
	enabledExtensionCount = 0;


	/*
		Stepping through each extension, we check for an extension named
		"VK_KHR_swapchain". This is stored in the token VK_KHR_SWAPCHAIN_EXTENSION_NAME.
	*/
	for (uint32_t i = 0; i < deviceExtensionCount; ++i){

		if (!strcmp(VK_KHR_SWAPCHAIN_EXTENSION_NAME, deviceExtensions[i].extensionName)){
			/*
				If and when we find the extension, we add it to the
				array of extension names that we declared at the top
				of this method and increment the number of extensions
				that we require.
			*/
			swapChainFound = 1;
			extension_names[enabledExtensionCount++] = VK_KHR_SWAPCHAIN_EXTENSION_NAME;
		}
		assert(enabledExtensionCount < 64);
	}


	/*
		If we didn't find the VK_KHR_swapchain extension, then
		we can't display to the screen. For the purposes of the
		tutorial there would be no point in continuing.
	*/
	if (!swapChainFound){
#if defined(WIN32)
		MessageBoxA(NULL, "Could not find swap chain device extension", "Error", MB_OK);
#else
		printf("Could not find swap chain device extension.\n");
#endif
		exit(1);
	}


	vkGetPhysicalDeviceProperties(
		m_physical_device,
		&m_device_properties
		);

    m_queue_count = 4;


    vkGetPhysicalDeviceQueueFamilyProperties(
		m_physical_device,
        &m_queue_count,
		NULL
		);

	m_queue_properties = (VkQueueFamilyProperties*)malloc(sizeof(VkQueueFamilyProperties) * m_queue_count);

	vkGetPhysicalDeviceMemoryProperties(
		m_physical_device,
		&m_memory_properties
		);


	vkGetPhysicalDeviceQueueFamilyProperties(
		m_physical_device,
		&m_queue_count,
		m_queue_properties
		);

	for (uint32_t i = 0; i < m_queue_count; ++i){
		VkQueueFamilyProperties qfp = m_queue_properties[i];
		continue;
	}

	VkDeviceQueueCreateInfo queueInfo;
	float qp[] = {1.0};
	queueInfo.queueCount = m_queue_count;
	queueInfo.queueFamilyIndex = 0;
	queueInfo.pQueuePriorities = qp;
	m_extension_count = 0;
	//m_extension_names = NULL;

    deviceCreate(&m_device, &m_physical_device, m_queue_count, &queueInfo, enabledExtensionCount, extension_names);


    std::cout << "Device ID : " << m_device << std::endl;



}



VulkanDC::Device::~Device(){}

VulkanDC *VulkanDC::Get(){
	static VulkanDC *outContext = NULL;

	if (!outContext){
		outContext = new VulkanDC();
	}
	return outContext;
}

VulkanDC::Device *VulkanDC::getDevice(uint32_t inDeviceID){
	return m_devices[inDeviceID];
}



VulkanDC::Device::Queue *VulkanDC::Device::newQueue(VulkanDC::Device::Queue::Name &inName, VulkanDC::Device::Queue::NodeID &inID){
	VulkanDC::Device::Queue *outQueue = new VulkanDC::Device::Queue(inName, inID);
	outQueue->initQueue(this);
	m_queues[inName] = outQueue;
	return outQueue;
}


VulkanDC::VulkanDC() :
m_has_instance(false),
m_default_device(NULL)
{
}

void VulkanDC::initDC(const VkInstance &inInstance){
	if (inInstance == VK_NULL_HANDLE) return;
	m_vk_instance = inInstance;
	m_has_instance = true;

	initDevices();
}

uint32_t VulkanDC::getQueueCount(uint32_t inDeviceID){
	VulkanDC::Device *device = getDevice(inDeviceID);
	if (!device) return 0;
	return device->getQueueCount();
}



void VulkanDC::initDevices(){
	if (!m_has_instance) return;

    std::cout << "Initialising Devices" << std::endl;

	m_device_count = 0;
	
	VKA_CHECK_ERROR(vkEnumeratePhysicalDevices(
		m_vk_instance,
		&m_device_count,
		NULL
		),"Could not get physical device count.\n");


    std::cout << "Found " << m_device_count << " Devices" << std::endl;

	m_physical_devices = (VkPhysicalDevice*)malloc(sizeof(VkPhysicalDevice) * m_device_count);

	VKA_CHECK_ERROR(vkEnumeratePhysicalDevices(
		m_vk_instance,
		&m_device_count,
		m_physical_devices
		),"Could not get physical devices.\n");

	for (uint32_t i = 0; i < m_device_count; ++i){
		m_default_device = newDevice(m_physical_devices[i]);
		if(m_default_device) break;
	}
}

VulkanDC::Device::Queue *VulkanDC::Device::createGraphicsQueue(VulkanDC::Device::Queue::Name &inName,VulkanDC::Device::Queue::NodeID inID){

    return newQueue(inName,m_graphics_queue_index);
}

VulkanDC::Device::Queue *VulkanDC::getQueueForGraphics(VulkanDC::Device::Queue::Name &queueName, VkFormat &outFormat,VulkanDC::Device::Queue::NodeID inID){
    VulkanDC::Device::Queue *queue = NULL;

    for(uint32_t dc = 0;dc<m_device_count;++dc){
        VulkanDC::Device *device = getDevice(dc);

        queue = device->createGraphicsQueue(queueName,inID);
        if(queue){

            outFormat = device->getFormat();
            return queue;
        }
    }

    return NULL;
}

VulkanDC::Device *VulkanDC::newDevice(const VkPhysicalDevice &inDevice){

	VulkanDC::Device *outDevice = new VulkanDC::Device(inDevice);
    m_default_device = outDevice;
	m_devices.push_back(outDevice);
	return outDevice;
}

VulkanDC::~VulkanDC()
{
}
